/*
 * Decompiled with CFR 0.152.
 */
package linkchecker;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.net.URISyntaxException;
import java.nio.charset.StandardCharsets;
import java.util.Collection;
import java.util.Optional;
import java.util.Scanner;
import java.util.logging.Level;
import java.util.regex.Pattern;
import linkchecker.Issue;
import linkchecker.Severity;
import linkchecker.URLProcessor;
import linkchecker.WebResource;
import linkchecker.interfaces.ContentFilter;
import linkchecker.interfaces.URLFilter;
import linkchecker.lib.StringUtil;

final class URLProcessingThread
extends Thread {
    private static final Pattern URI_PATTERN = Pattern.compile("(?:https?://[^\\s\"'><]+| (?:href|data-uri|data-url|url|uri|src|ref)=\"[^\"]+\"|url\\([^d][^)]+\\);)", 2);
    private final URLProcessor processor;
    private final Collection<URLFilter> urlFilters;
    private final Collection<ContentFilter> contentFilters;

    URLProcessingThread(URLProcessor processor, int id, Collection<URLFilter> urlFilters, Collection<ContentFilter> contentFilters) {
        super((ThreadGroup)processor, "Resolver " + id);
        this.setDaemon(true);
        this.processor = processor;
        this.urlFilters = urlFilters;
        this.contentFilters = contentFilters;
    }

    @Override
    public void run() {
        while (!this.isInterrupted()) {
            try {
                WebResource currentLink = this.processor.poll();
                if (currentLink == null) break;
                this.lookup(currentLink);
            }
            catch (InterruptedException currentLink) {
            }
            catch (Throwable ex) {
                URLProcessor.LOGGER.log(Level.SEVERE, "Error in main processing loop", ex);
            }
        }
    }

    private void lookup(WebResource resource) {
        boolean follow;
        Optional<InputStream> stream = resource.fetch();
        URI location = resource.getLocation();
        String mime = resource.getContentType();
        if (mime == null || !stream.isPresent()) {
            return;
        }
        if (location != null && (follow = this.urlFilters.stream().allMatch(filter -> filter.shouldFollow(resource, location)))) {
            this.processor.submit(location, resource);
        }
        try (InputStream dataStream = stream.get();){
            byte[] contents = URLProcessingThread.readContents(dataStream);
            if (mime.startsWith("text/") || mime.contains("xml")) {
                try (ByteArrayInputStream in = new ByteArrayInputStream(contents);){
                    this.followLinks(mime, in, resource);
                }
                catch (IOException ex) {
                    URLProcessor.LOGGER.log(Level.SEVERE, "Error following links", ex);
                }
            }
            this.contentFilters.forEach(filter -> URLProcessingThread.dispatchToContentFilters(resource, contents, filter));
        }
        catch (IOException ex) {
            resource.addFatalIssue("IO Error", "Error reading contents: " + ex.getMessage());
        }
    }

    private static void dispatchToContentFilters(WebResource resource, byte[] contents, ContentFilter filter) {
        try (ByteArrayInputStream in = new ByteArrayInputStream(contents);){
            filter.examineContents(in, resource);
        }
        catch (Throwable ex) {
            resource.addFatalIssue("Error in Content Filter", "Error running content filter " + filter.getClass().getName() + ": " + ex.getMessage());
            URLProcessor.LOGGER.log(Level.SEVERE, "Error in content filter " + filter, ex);
        }
    }

    private static byte[] readContents(InputStream inputStream) throws IOException {
        try (ByteArrayOutputStream result = new ByteArrayOutputStream();){
            int length;
            byte[] buffer = new byte[4096];
            while ((length = inputStream.read(buffer)) != -1) {
                result.write(buffer, 0, length);
            }
            inputStream.close();
            byte[] byArray = result.toByteArray();
            return byArray;
        }
    }

    void followLinks(String mime, InputStream in, WebResource resource) {
        URI current = URLProcessingThread.normaliseCurrentUrl(resource);
        try (Scanner scanner = new Scanner(in, StandardCharsets.UTF_8.name());){
            String potential;
            boolean isXml;
            boolean bl = isXml = mime.contains("xml") || mime.contains("text/html");
            while ((potential = scanner.findWithinHorizon(URI_PATTERN, 0)) != null) {
                if ((potential = URLProcessingThread.cleanUriString(potential, isXml)).startsWith("data:") || potential.startsWith("mailto:")) continue;
                try {
                    URI next = current.resolve(potential);
                    if (this.urlFilters.stream().allMatch(filter -> filter.shouldFollow(resource, next))) {
                        this.processor.submit(next, resource);
                        continue;
                    }
                    resource.addIssue(new Issue(Severity.IGNORE, URLFilter.class, "Link not followed", potential, 0, 0));
                }
                catch (IllegalArgumentException ex) {
                    resource.addIssue(new Issue(Severity.INFO, URLFilter.class, "Link not understood", potential + ": " + ex, 0, 0));
                }
            }
        }
    }

    private static String cleanUriString(String potential, boolean isXml) {
        String parsed = potential;
        if (isXml) {
            parsed = StringUtil.unEscapeHtml(parsed).toString();
        }
        if (parsed.endsWith(");")) {
            parsed = parsed.substring(parsed.indexOf(40) + 1, parsed.length() - 2);
        } else if (parsed.endsWith(")")) {
            parsed = parsed.substring(parsed.indexOf(40) + 1, parsed.length() - 1);
        }
        if (parsed.endsWith("\\\"")) {
            parsed = parsed.substring(0, parsed.length() - 2);
            parsed = parsed.substring(parsed.indexOf("\\\"") + 2);
        }
        if (parsed.endsWith("\"")) {
            parsed = parsed.substring(0, parsed.length() - 1);
            parsed = parsed.substring(parsed.indexOf(34) + 1);
        }
        return parsed;
    }

    private static URI normaliseCurrentUrl(WebResource resource) {
        URI current = resource.getUri();
        if (current.getPath().isEmpty()) {
            current = current.resolve("/");
        }
        if (current.getFragment() != null) {
            String noFrag = current.toString();
            noFrag = noFrag.substring(0, noFrag.indexOf(35));
            try {
                current = new URI(noFrag);
            }
            catch (URISyntaxException ex) {
                URLProcessor.LOGGER.log(Level.SEVERE, "Error normalising URL", ex);
            }
        }
        return current;
    }
}

